// ===== GAME DATA =====
const games = [
    { name: "Gates of Olympus 1000", image: "olympus1000.webp", rtp: 96, provider: "Pragmatic Play" },
    { name: "Mahjong Ways", image: "mahjongways.webp", rtp: 95, provider: "PG Soft" },
    { name: "Starlight Princess", image: "incess.webp", rtp: 89, provider: "Pragmatic Play" },
    { name: "Gates of Olympus", image: "olympus.webp", rtp: 97, provider: "Pragmatic Play" },
    { name: "Mahjong Ways 2", image: "mahjongways2.webp", rtp: 92, provider: "PG Soft" },
    { name: "Starlight Princess 1000", image: "incess1000.webp", rtp: 91, provider: "Pragmatic Play" },
    { name: "5 Lions Reborn", image: "5-lions-reborn.webp", rtp: 65, provider: "Pragmatic Play" },
    { name: "Assgardian Rissing", image: "asgardian-rissing.webp", rtp: 68, provider: "Pragmatic Play" },
    { name: "Aztec Gems", image: "aztecgemz.webp", rtp: 88, provider: "Pragmatic Play" },
    { name: "Bang Gacor 1000", image: "bang-gacor-1000.webp", rtp: 92, provider: "Pragmatic Play" },
    { name: "Sweet Bonanza 1000", image: "bonanza-1000.webp", rtp: 90, provider: "Pragmatic Play" },
    { name: "Buffalo Win", image: "buffalowin.webp", rtp: 67, provider: "Pragmatic Play" },
    { name: "Caishen Wins", image: "cai-shen-wins.webp", rtp: 79, provider: "Pragmatic Play" },
    { name: "Candy Bonanza", image: "candy-bonanza.webp", rtp: 97, provider: "Pragmatic Play" },
    { name: "Candy Brust", image: "candy-burst.webp", rtp: 92, provider: "Pragmatic Play" },
    { name: "Captain Bounty", image: "captain-bounty.webp", rtp: 93, provider: "Pragmatic Play" },
    { name: "Cocktail Nights", image: "cocktailnights.webp", rtp: 85, provider: "Pragmatic Play" },
    { name: "Dragon Hatch", image: "dragon-hatch.webp", rtp: 90, provider: "Pragmatic Play" },
    { name: "Dragon Hatch 2", image: "dragonhatch2.webp", rtp: 82, provider: "Pragmatic Play" },
    { name: "Dragon Tiger Luck", image: "dragon-tiger-luck.webp", rtp: 92, provider: "Pragmatic Play" },
    { name: "Dreams of Macau", image: "dreams-of-macau.webp", rtp: 84, provider: "Pragmatic Play" },
    { name: "Fortune Dragon", image: "fortunedragon.webp", rtp: 90, provider: "Pragmatic Play" },
    { name: "Fortune Rabbit", image: "fortune-rabbit.webp", rtp: 66, provider: "Pragmatic Play" },
    { name: "Fortune Tiger", image: "fortunetiger.webp", rtp: 94, provider: "Pragmatic Play" },
    { name: "Fortune Tree", image: "fortunetree.webp", rtp: 82, provider: "Pragmatic Play" },
    { name: "Fruity Candy", image: "fruitycandy.webp", rtp: 66, provider: "Pragmatic Play" },
    { name: "Ganesha Fortune", image: "ganesha-fortune.webp", rtp: 84, provider: "Pragmatic Play" },
    { name: "Garuda Gems", image: "garuda-gems.webp", rtp: 75, provider: "Pragmatic Play" },
    { name: "Gemstones Gold", image: "gemstonesgold.webp", rtp: 83, provider: "Pragmatic Play" },
    { name: "Jurrasic Kingdom", image: "jurassickdm.webp", rtp: 65, provider: "Pragmatic Play" },
    { name: "Lucky Neko", image: "luckyneko.webp", rtp: 76, provider: "Pragmatic Play" },
    { name: "Lucky Piggy", image: "luckypiggy.webp", rtp: 68, provider: "Pragmatic Play" },
    { name: "Mafia Mayhem", image: "mafiamahyem.webp", rtp: 69, provider: "Pragmatic Play" },
    { name: "Mahjong Wins 3", image: "mahjong-wins-3.webp", rtp: 71, provider: "PG Soft" },
    { name: "Mahjong Wins 2", image: "mahwin2.webp", rtp: 65, provider: "PG Soft" },
    { name: "Majestic Treasures", image: "majestic-ts.webp", rtp: 70, provider: "Pragmatic Play" },
    { name: "Midas Fortune", image: "midas-fortune.webp", rtp: 73, provider: "Pragmatic Play" },
    { name: "Legendary Monkey King", image: "monkeyking.webp", rtp: 87, provider: "Pragmatic Play" },
    { name: "Phoenix Rises", image: "phoenix-rises.webp", rtp: 72, provider: "Pragmatic Play" },
    { name: "Pinata Wins", image: "pinatawins.webp", rtp: 70, provider: "Pragmatic Play" },
    { name: "Pyramid Bonanza", image: "pyramidbonanza.webp", rtp: 69, provider: "Pragmatic Play" },
    { name: "Queen of Bounty", image: "queen-bounty.webp", rtp: 94, provider: "Pragmatic Play" },
    { name: "Rooster Rumble", image: "roosterrumble.webp", rtp: 68, provider: "Pragmatic Play" },
    { name: "Speed Winner", image: "speedwinner.webp", rtp: 67, provider: "Pragmatic Play" },
    { name: "Spirited Wonders", image: "spirited-wonders.webp", rtp: 92, provider: "Pragmatic Play" },
    { name: "The Great Icescape", image: "thegreaticescape.webp", rtp: 97, provider: "Pragmatic Play" },
    { name: "Tottem Wonders", image: "tottemwonders.webp", rtp: 90, provider: "Pragmatic Play" },
    { name: "Treasures Aztec", image: "treasuresaztec.webp", rtp: 76, provider: "Pragmatic Play" },
    { name: "Ultimate Striker", image: "ultimate-striker.webp", rtp: 88, provider: "Pragmatic Play" },
    { name: "888 Gold", image: "vs5triple8gold.webp", rtp: 88, provider: "Pragmatic Play" },
    { name: "Big Bass Mission Fishing", image: "vs10bbfmission.webp", rtp: 96, provider: "Pragmatic Play" },
    { name: "Joker's Jewels Hot", image: "vs10jokerhot.webp", rtp: 91, provider: "Pragmatic Play" },
    { name: "Candy Village", image: "vs20candvil.webp", rtp: 84, provider: "Pragmatic Play" },
    { name: "Colossal Cash Zone", image: "vs20colcashzone.webp", rtp: 78, provider: "Pragmatic Play" },
    { name: "Gates of Olympus 1000 Dice", image: "vs20dicegatex.webp", rtp: 72, provider: "Pragmatic Play" },
    { name: "Nexus Sweet Bonanza", image: "vs20fruitsnexus.webp", rtp: 67, provider: "Pragmatic Play" },
    { name: "Gates of Gatot Kaca", image: "vs20gatotgates.webp", rtp: 74, provider: "Pragmatic Play" },
    { name: "Gates of Gatot Kaca 1000", image: "vs20gatotx.webp", rtp: 86, provider: "Pragmatic Play" },
    { name: "Nexus Gates of Olympus", image: "vs20nexusgates.webp", rtp: 74, provider: "Pragmatic Play" },
    { name: "Starlight Christmas", image: "vs20schristmas.webp", rtp: 86, provider: "Pragmatic Play" },
    { name: "Slot88 Rush", image: "vs20slot88sr.webp", rtp: 95, provider: "Pragmatic Play" },
    { name: "Sugar Rush", image: "vs20sugarrush.webp", rtp: 72, provider: "Pragmatic Play" },
    { name: "Sugar Rush Xmas", image: "vs20sugrux.webp", rtp: 72, provider: "Pragmatic Play" },
    { name: "Sweet Bonanza 1000 Dice", image: "vs20swdicex.webp", rtp: 67, provider: "Pragmatic Play" },
    { name: "Cash Patrol", image: "vs25copsrobbers.webp", rtp: 86, provider: "Pragmatic Play" },
    { name: "Hokkaido Wolf", image: "vs576hokkwolf.webp", rtp: 92, provider: "Pragmatic Play" },
    { name: "5 Lions Megaways 2", image: "vsways5lions2.webp", rtp: 91, provider: "Pragmatic Play" },
    { name: "Aztec Gems Megaways", image: "vswaysaztec.webp", rtp: 78, provider: "Pragmatic Play" },
    { name: "The Dog House Megaways", image: "vswaysdogs.webp", rtp: 94, provider: "Pragmatic Play" },
    { name: "Power of Thor Megaways", image: "vswayshammthor.webp", rtp: 73, provider: "Pragmatic Play" },
    { name: "Rise of Samurai Megaways", image: "vswayssamurai.webp", rtp: 81, provider: "Pragmatic Play" },
    { name: "Wild West Gold Megaways", image: "vswayswildwest.webp", rtp: 70, provider: "Pragmatic Play" },
    { name: "Ways of the Qilin", image: "waysofqilin.webp", rtp: 65, provider: "Pragmatic Play" },
    { name: "Wild Bandito", image: "wildbandito.webp", rtp: 97, provider: "Pragmatic Play" },
    { name: "Wild Bounty Showdown", image: "wildbounty.webp", rtp: 95, provider: "Pragmatic Play" },
    { name: "Blood & Shadow", image: "bloodandshadow.webp", rtp: 71, provider: "Pragmatic Play" },
    { name: "Blood & Shadow 2", image: "bloodandshadow2.webp", rtp: 98, provider: "Pragmatic Play" },
    { name: "Boxing Riches", image: "boxing-riches.webp", rtp: 73, provider: "Pragmatic Play" },
    { name: "Buffalo Hunter", image: "buffalohunter.webp", rtp: 98, provider: "Pragmatic Play" },
    { name: "Gluttony", image: "gluttony.webp", rtp: 85, provider: "Pragmatic Play" },
    { name: "Highway King", image: "Highway King.webp", rtp: 76, provider: "Pragmatic Play" },
    { name: "Just Jewels", image: "Just-Jewels.webp", rtp: 85, provider: "Pragmatic Play" },
    { name: "Merge Magic", image: "merge-magic.webp", rtp: 91, provider: "Pragmatic Play" },
    { name: "Pirate-King", image: "pirate-king.webp", rtp: 94, provider: "Pragmatic Play" },
    { name: "Qixi", image: "Qixi.webp", rtp: 80, provider: "Pragmatic Play" },
    { name: "Queen of Inca", image: "queen-of-inca.webp", rtp: 95, provider: "Pragmatic Play" },
    { name: "Roma Gladiatrix", image: "roma-gladiatrix.webp", rtp: 67, provider: "Pragmatic Play" },
    { name: "Serial", image: "serial.webp", rtp: 66, provider: "Pragmatic Play" },
    { name: "5 Lucky Lions", image: "SG5LuckyLions.webp", rtp: 68, provider: "Pragmatic Play" },
    { name: "Arctic Hunt", image: "SGArcticHunt.webp", rtp: 66, provider: "Pragmatic Play" },
    { name: "Fa Cai Shen", image: "SGFaCaiShen.webp", rtp: 70, provider: "Pragmatic Play" },
    { name: "Koi Koi Treasure", image: "SGKoiKoiTreasure.webp", rtp: 71, provider: "Pragmatic Play" },
    { name: "Laughing Buddha", image: "SGLaughingBuddha.webp", rtp: 96, provider: "Pragmatic Play" },
    { name: "Nexus Koi Gate", image: "SGNexusKoiGate.webp", rtp: 72, provider: "Pragmatic Play" },
    { name: "Nexus Wealth Inn", image: "SGNexusWealthInn.webp", rtp: 73, provider: "Pragmatic Play" },
    { name: "The Koi Gate", image: "SGTheKoiGate.webp", rtp: 78, provider: "Pragmatic Play" },
    { name: "Tuk Tuk Thailand", image: "SGTukTukThailand.webp", rtp: 90, provider: "Pragmatic Play" },
    { name: "Zeus Deluxe", image: "SGZeusDeluxe.webp", rtp: 94, provider: "Pragmatic Play" },
    { name: "Sugar Bang Bang", image: "sugar-bang-bang.webp", rtp: 90, provider: "Pragmatic Play" },
    { name: "Totem Island", image: "totem-island.webp", rtp: 72, provider: "Pragmatic Play" }
];

// ===== UTILITY FUNCTIONS =====
const getColor = (rtp) => {
    if (rtp >= 90) return "#2ecc71"; // hijau untuk RTP tinggi
    if (rtp >= 80) return "#f1c40f"; // kuning untuk RTP sedang
    if (rtp >= 70) return "#e67e22"; // oranye untuk RTP rendah
    return "#e74c3c"; // merah untuk RTP sangat rendah
};

const generateJamGacor = () => {
    const startHour = Math.floor(Math.random() * 22);
    const endHour = startHour + 2;
    const format = (h) => String(h).padStart(2, "0") + ":00";
    return `${format(startHour)} - ${format(endHour)}`;
};

// ===== DOM ELEMENTS =====
const gameGrid = document.getElementById("gameGrid");
const searchInput = document.getElementById("searchGame");
const btnSemua = document.getElementById("btnSemua");
const btnPopuler = document.getElementById("btnPopuler");
const btnWinrate = document.getElementById("btnWinrate");
const hamburger = document.getElementById('hamburger');
const navMenu = document.getElementById('navMenu');
const closeBtn = document.getElementById('closeBtn');
const darkToggle = document.getElementById('darkModeToggle');
const scrollBtn = document.getElementById("scrollToTopBtn");

// ===== GAME RENDERING =====
function renderGames(gameList) {
    gameGrid.innerHTML = "";
    
    if (gameList.length === 0) {
        gameGrid.innerHTML = `
            <div class="no-games-found">
                <i class="fas fa-search"></i>
                <h3>Game tidak ditemukan</h3>
                <p>Coba kata kunci lain atau filter berbeda</p>
            </div>
        `;
        return;
    }

    gameList.forEach((game, index) => {
        const card = document.createElement("div");
        card.className = "game-card fade-in";
        
        const jamGacor = generateJamGacor();

        card.innerHTML = `
            <div class="game-image-wrapper">
                <img src="assets/${game.image}" alt="${game.name}" class="game-image" onerror="this.src='https://via.placeholder.com/300x180/1a2540/ffffff?text=Game+Image'">
                <a href="https://wk168.pro/sgk" class="play-btn">
                    <i class="fas fa-play"></i>
                    MAIN SEKARANG
                </a>
            </div>
            <div class="game-content">
                <h3 class="game-title">${game.name}</h3>
                <div class="game-info">
                    <span class="game-provider">${game.provider || 'Pragmatic Play'}</span>
                    <span class="rtp-badge">RTP ${game.rtp}%</span>
                </div>
                <div class="rtp-bar">
                    <div class="rtp-bar-fill" style="width: ${game.rtp}%; background-color: ${getColor(game.rtp)}">
                        <div class="warp-bg"></div>
                        <span class="rtp-value">${game.rtp}%</span>
                    </div>
                </div>
                <div class="jam-gacor-slot">
                    <i class="fas fa-clock"></i>
                    JAM GACOR: <strong>${jamGacor}</strong>
                </div>
            </div>
        `;

        gameGrid.appendChild(card);
    });
}

function getRTPClass(rtp) {
    if (rtp >= 95) return 'high-rtp';
    if (rtp >= 90) return 'medium-rtp';
    return 'low-rtp';
  }

// ===== FILTER FUNCTIONS =====
btnSemua.addEventListener('click', () => {
    setActiveFilter(btnSemua);
    const shuffledGames = [...games].sort(() => 0.5 - Math.random()).slice(0, 50);
    renderGames(shuffledGames);
});

btnPopuler.addEventListener('click', () => {
    setActiveFilter(btnPopuler);
    const populerGames = games.filter(g => 
        /Olympus|Mahjong|Princess|Bonanza|Fortune|Dragon|Tiger/i.test(g.name)
    );
    renderGames(populerGames);
});

btnWinrate.addEventListener('click', () => {
    setActiveFilter(btnWinrate);
    const highWinrate = games.filter(g => g.rtp >= 90);
    renderGames(highWinrate);
});

searchInput.addEventListener("input", (e) => {
    const keyword = e.target.value.toLowerCase();
    const result = games.filter(g => 
        g.name.toLowerCase().includes(keyword) || 
        (g.provider && g.provider.toLowerCase().includes(keyword))
    );
    renderGames(result);
});

function setActiveFilter(activeBtn) {
    document.querySelectorAll('.filter-btn').forEach(btn => {
        btn.classList.remove('active');
    });
    activeBtn.classList.add('active');
}

// ===== RTP UPDATE SYSTEM =====
function animateRTPValue(start, end, duration, onUpdate) {
    const startTime = performance.now();

    function animate(currentTime) {
        const elapsed = currentTime - startTime;
        const progress = Math.min(elapsed / duration, 1);
        const currentValue = Math.round(start + (end - start) * progress);
        onUpdate(currentValue);

        if (progress < 1) {
            requestAnimationFrame(animate);
        }
    }

    requestAnimationFrame(animate);
}

function updateRTP() {
    const bars = document.querySelectorAll(".rtp-bar-fill");
    const values = document.querySelectorAll(".rtp-value");

    bars.forEach((bar, i) => {
        const text = values[i];
        const current = parseInt(text.textContent.replace('%', '')) || 0;
        const newRtp = Math.floor(Math.random() * (98 - 65 + 1)) + 65;

        animateRTPValue(current, newRtp, 1000, (val) => {
            text.textContent = val + "%";
            bar.style.width = val + "%";
            bar.style.backgroundColor = getColor(val);
            
            // Update RTP badge juga
            const gameCard = bar.closest('.game-card');
            if (gameCard) {
                const rtpBadge = gameCard.querySelector('.rtp-badge');
                if (rtpBadge) {
                    rtpBadge.textContent = `RTP ${val}%`;
                }
            }
        });
    });
}

// ===== NAVIGATION & UI CONTROLS =====
hamburger.addEventListener('click', () => {
    navMenu.classList.add('active');
    hamburger.classList.add('active');
    document.body.style.overflow = 'hidden';
});

closeBtn.addEventListener('click', () => {
    navMenu.classList.remove('active');
    hamburger.classList.remove('active');
    document.body.style.overflow = '';
});

// Dark mode toggle
darkToggle.addEventListener('change', () => {
    document.body.setAttribute('data-theme', darkToggle.checked ? 'dark' : 'light');
    localStorage.setItem('theme', darkToggle.checked ? 'dark' : 'light');
});

// Scroll to top
window.addEventListener("scroll", () => {
    if (window.scrollY > 200) {
        scrollBtn.classList.add('show');
    } else {
        scrollBtn.classList.remove('show');
    }
});

scrollBtn.addEventListener("click", () => {
    window.scrollTo({ top: 0, behavior: "smooth" });
});

// ===== DATE & TIME UPDATES =====
function updateDateTime() {
    const hari = ["Minggu", "Senin", "Selasa", "Rabu", "Kamis", "Jumat", "Sabtu"];
    const bulan = [
        "Januari", "Februari", "Maret", "April", "Mei", "Juni",
        "Juli", "Agustus", "September", "Oktober", "November", "Desember"
    ];

    const now = new Date();
    const tanggal = `${hari[now.getDay()]}, ${now.getDate()} ${bulan[now.getMonth()]} ${now.getFullYear()}`;
    const waktu = now.toLocaleTimeString("id-ID", { 
        hour: "2-digit", 
        minute: "2-digit", 
        second: "2-digit" 
    });
    
    const rtpEl = document.getElementById("rtpTanggal");
    if (rtpEl) rtpEl.textContent = `Update RTP: ${tanggal} | ${waktu}`;
}

// ===== PARTICLE BACKGROUND =====
function createParticles() {
    const particlesContainer = document.getElementById('particles');
    const particleCount = 50;

    for (let i = 0; i < particleCount; i++) {
        const particle = document.createElement('div');
        particle.className = 'particle';
        
        // Random position
        particle.style.left = Math.random() * 100 + 'vw';
        particle.style.top = Math.random() * 100 + 'vh';
        
        // Random size
        const size = Math.random() * 3 + 1;
        particle.style.width = size + 'px';
        particle.style.height = size + 'px';
        
        // Random animation delay and duration
        particle.style.animationDelay = Math.random() * 6 + 's';
        particle.style.animationDuration = (Math.random() * 10 + 10) + 's';
        
        particlesContainer.appendChild(particle);
    }
}

// ===== FLOATING PLAYER COUNTER =====
class FloatingPlayerCounter {
    constructor() {
        this.minPlayers = 500;
        this.maxPlayers = 2500;
        this.currentPlayers = 0;
        this.updateInterval = null;
        this.isInitialized = false;
    }

    init() {
        if (this.isInitialized) return;
        
        this.currentPlayers = this.generatePlayerCount();
        this.updateDisplay();
        this.startAutoUpdate();
        this.isInitialized = true;
        
        // Close button functionality
        const closeBtn = document.getElementById('floatingClose');
        if (closeBtn) {
            closeBtn.addEventListener('click', () => {
                this.hideCounter();
            });
        }
    }

    generatePlayerCount() {
        const baseCount = Math.floor(Math.random() * (this.maxPlayers - this.minPlayers)) + this.minPlayers;
        const now = new Date();
        const hour = now.getHours();
        
        let timeMultiplier = 1;
        if (hour >= 18 || hour <= 6) {
            timeMultiplier = 1.3; // Peak hours
        } else if (hour >= 12 && hour <= 14) {
            timeMultiplier = 1.1; // Lunch time
        }
        
        return Math.floor(baseCount * timeMultiplier);
    }

    updateDisplay() {
        const playerCountElements = document.querySelectorAll('#playerCount');
        if (playerCountElements.length > 0) {
            const formattedCount = this.currentPlayers.toLocaleString();
            playerCountElements.forEach(element => {
                element.textContent = formattedCount;
            });
        }
    }

    updatePlayerCount() {
        const newTarget = this.generatePlayerCount();
        const difference = newTarget - this.currentPlayers;
        const step = Math.sign(difference) * Math.ceil(Math.abs(difference) / 10);
        
        this.animateCountChange(step, newTarget);
    }

    animateCountChange(step, target) {
        const animate = () => {
            this.currentPlayers += step;
            
            if ((step > 0 && this.currentPlayers >= target) || 
                (step < 0 && this.currentPlayers <= target)) {
                this.currentPlayers = target;
                this.updateDisplay();
                return;
            }
            
            this.updateDisplay();
            setTimeout(animate, 50);
        };
        
        animate();
    }

    startAutoUpdate() {
        this.updateInterval = setInterval(() => {
            this.updatePlayerCount();
        }, 30000); // Update every 30 seconds
    }

    hideCounter() {
        const counter = document.querySelector('.floating-player-counter');
        if (counter) {
            counter.style.display = 'none';
        }
    }

    destroy() {
        if (this.updateInterval) {
            clearInterval(this.updateInterval);
        }
        this.isInitialized = false;
    }
}

// ===== INITIALIZATION =====
document.addEventListener('DOMContentLoaded', function() {
    // Initialize theme
    const savedTheme = localStorage.getItem('theme') || 'light';
    document.body.setAttribute('data-theme', savedTheme);
    darkToggle.checked = savedTheme === 'dark';

    // Render initial games
    renderGames(games.slice(0, 50));

    // Initialize systems
    updateDateTime();
    setInterval(updateDateTime, 1000);
    
    createParticles();
    
    // Initialize player counter
    window.floatingPlayerCounter = new FloatingPlayerCounter();
    window.floatingPlayerCounter.init();

    // Start RTP updates
    setInterval(updateRTP, 10000);

    // Update RTP stats in header
    updateRTPStats();

    console.log('SAMOSIR88 RTP Platform initialized successfully');
});

// Update RTP statistics
function updateRTPStats() {
    const totalGames = games.length;
    const averageRTP = Math.round(games.reduce((sum, game) => sum + game.rtp, 0) / totalGames);
    const highestRTP = Math.max(...games.map(game => game.rtp));

    document.getElementById('rtpGames').textContent = totalGames;
    document.getElementById('rtpAverage').textContent = averageRTP + '%';
    document.getElementById('rtpHigh').textContent = highestRTP + '%';
}

// Enhanced RTP update with player activity
function enhanceRTPUpdateWithPlayerActivity() {
    const originalUpdateRTP = updateRTP;
    
    updateRTP = function() {
        originalUpdateRTP();
        
        if (window.floatingPlayerCounter && Math.random() > 0.7) {
            const smallChange = Math.floor(Math.random() * 21) - 10;
            window.floatingPlayerCounter.currentPlayers += smallChange;
            
            window.floatingPlayerCounter.currentPlayers = Math.max(
                window.floatingPlayerCounter.minPlayers,
                Math.min(window.floatingPlayerCounter.maxPlayers, window.floatingPlayerCounter.currentPlayers)
            );
            
            window.floatingPlayerCounter.updateDisplay();
        }
    };
}

enhanceRTPUpdateWithPlayerActivity();